'use strict';

/* --------------------------------------------------------------
 bulk_email_invoice.js 2016-10-05
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Bulk Email Invoice Modal Controller
 */
gx.controllers.module('bulk_email_invoice', ['modal', 'loading_spinner'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {
        bindings: { subject: $this.find('.subject') }
    };

    /**
     * Selector for the email list item.
     *
     * @type {String}
     */
    var emailListItemSelector = '.email-list-item';

    /**
     * Selector for the email list item ID.
     *
     * @type {String}
     */
    var emailListItemEmailSelector = '.email-input';

    /**
     * Selector for the modal content body layer.
     *
     * @type {String}
     */
    var modalContentSelector = '.modal-content';

    /**
     * Request URL
     *
     * @type {String}
     */
    var requestUrl = jse.core.config.get('appUrl') + '/admin/gm_pdf_order.php';

    /**
     * Loading spinner instance.
     *
     * @type {jQuery|null}
     */
    var $spinner = null;

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Show/hide loading spinner.
     *
     * @param {Boolean} doShow Show the loading spinner?
     */
    function _toggleSpinner(doShow) {
        if (doShow) {
            $spinner = jse.libs.loading_spinner.show($this.find(modalContentSelector), $this.css('z-index'));
        } else {
            jse.libs.loading_spinner.hide($spinner);
        }
    }

    /**
     * Parse subject and replace the placeholders with the variables.
     *
     * @param {Object} invoiceData Invoice data
     *
     * @return {String}
     */
    function _getParsedSubject(invoiceData) {
        var dateFormat = jse.core.config.get('languageCode') === 'de' ? 'DD.MM.YY' : 'MM.DD.YY';

        return module.bindings.subject.get().replace('{INVOICE_NUM}', invoiceData.invoiceNumber).replace('{INVOICE_DATE}', moment(invoiceData).format(dateFormat)).replace('{ORDER_ID}', invoiceData.orderId);
    }

    /**
     * Handles the successful delivery of all messages.
     */
    function _handleDeliverySuccess() {
        var message = jse.core.lang.translate('BULK_MAIL_SUCCESS', 'gm_send_order');

        // Show success message in the admin info box.
        jse.libs.info_box.addSuccessMessage(message);

        $('.invoices .table-main').DataTable().ajax.reload(null, false);
        $('.invoices .table-main').invoices_overview_filter('reload');

        // Hide modal and loading spinner.
        _toggleSpinner(false);
        $this.modal('hide');
    }

    /**
     * Handles the failure of the message delivery.
     */
    function _handleDeliveryFail() {
        var title = jse.core.lang.translate('error', 'messages');
        var message = jse.core.lang.translate('BULK_MAIL_UNSUCCESS', 'gm_send_order');

        // Show error message in a modal.
        jse.libs.modal.showMessage(title, message);

        // Hide modal and the loading spinner and re-enable the send button.
        _toggleSpinner(false);
        $this.modal('hide');
    }

    /**
     * Validate the form for empty fields.
     *
     * @return {Boolean}
     */
    function _validateForm() {
        $this.find('.has-error').removeClass('has-error');

        $this.find('input:text').each(function (index, input) {
            var $input = $(input);

            if ($input.val() === '') {
                $input.parents('.form-group').addClass('has-error');
            }
        });

        return !$this.find('.has-error').length;
    }

    /**
     * Send the modal data to the form through an AJAX call.
     */
    function _onSendClick() {
        if (!_validateForm()) {
            return;
        }

        // Collection of requests in promise format.
        var promises = [];

        // Email list item elements.
        var $emailListItems = $this.find(emailListItemSelector);

        // Abort and hide modal on empty email list entries.
        if (!$emailListItems.length) {
            $this.modal('hide');
            return;
        }

        // Show loading spinner.
        _toggleSpinner(true);

        // Fill orders array with data.
        $emailListItems.each(function (index, element) {
            // Order data.
            var invoiceData = $(element).data('invoice');

            // Email address entered in input field.
            var enteredEmail = $(element).find(emailListItemEmailSelector).val();

            // Promise wrapper for AJAX requests.
            var promise = new Promise(function (resolve, reject) {
                // Request GET parameters to send.
                var getParameters = {
                    oID: invoiceData.orderId,
                    iID: invoiceData.invoiceId,
                    type: 'invoice',
                    mail: '1',
                    gm_quick_mail: '1',
                    preview: '1'
                };

                // Composed request URL.
                var url = requestUrl + '?' + $.param(getParameters);

                // Data to send.
                var data = {
                    gm_mail: enteredEmail,
                    gm_subject: _getParsedSubject(invoiceData)
                };

                $.ajax({ method: 'POST', url: url, data: data }).done(resolve).fail(reject);
            });

            // Add promise to array.
            promises.push(promise);
        });

        // Wait for all promise to respond and handle success/error.
        Promise.all(promises).then(_handleDeliverySuccess).catch(_handleDeliveryFail);
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('click', '.btn.send', _onSendClick);
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
